const express = require('express');
const { createServer } = require('http');
const { Server } = require('socket.io');
const mysql = require('mysql2/promise');
const cors = require('cors');

const app = express();
app.use(cors());
app.use(express.json());

const httpServer = createServer(app);
const io = new Server(httpServer, {
  cors: {
    origin: "*",
    methods: ["GET", "POST"]
  }
});

// Database config
const dbConfig = {
  host: 'localhost',
  user: 'your_username',
  password: 'your_password',
  database: 'studypanda_db'
};

let db;

// Connect to database
async function connectDB() {
  try {
    db = await mysql.createPool(dbConfig);
    console.log('✅ Database connected');
  } catch (err) {
    console.error('❌ Database connection failed:', err);
    process.exit(1);
  }
}

// Store active matches and connections
const activeMatches = new Map(); // matchId => { players: [], scores: {} }
const userSockets = new Map();   // userId => socketId

// Socket.IO events
io.on('connection', (socket) => {
  console.log('🔌 New connection:', socket.id);
  
  // User authentication
  socket.on('auth', (data) => {
    const { userId, token } = data;
    // TODO: Verify JWT token
    socket.userId = userId;
    userSockets.set(userId, socket.id);
    
    socket.emit('auth_success', { userId });
    console.log(`✅ User ${userId} authenticated`);
  });
  
  // Join game match
  socket.on('join_match', async (data) => {
    const { matchId } = data;
    const userId = socket.userId;
    
    if (!userId) {
      socket.emit('error', { message: 'Not authenticated' });
      return;
    }
    
    // Join room
    socket.join(`match_${matchId}`);
    
    // Track match players
    if (!activeMatches.has(matchId)) {
      activeMatches.set(matchId, {
        players: [],
        scores: {},
        status: 'waiting'
      });
    }
    
    const match = activeMatches.get(matchId);
    if (!match.players.includes(userId)) {
      match.players.push(userId);
    }
    
    // Notify all players in match
    io.to(`match_${matchId}`).emit('player_joined', {
      userId,
      playerCount: match.players.length,
      players: match.players
    });
    
    console.log(`👥 User ${userId} joined match ${matchId}`);
  });
  
  // Submit answer
  socket.on('submit_answer', async (data) => {
    const { matchId, questionIndex, answer, isCorrect, points, timeTaken } = data;
    const userId = socket.userId;
    
    if (!activeMatches.has(matchId)) return;
    
    const match = activeMatches.get(matchId);
    
    // Update score
    if (!match.scores[userId]) {
      match.scores[userId] = 0;
    }
    match.scores[userId] += points;
    
    // Broadcast to all players in match
    io.to(`match_${matchId}`).emit('player_answered', {
      userId,
      questionIndex,
      isCorrect,
      points,
      timeTaken,
      currentScore: match.scores[userId]
    });
    
    // Save to database
    try {
      await db.query(
        `INSERT INTO match_answers 
         (match_id, user_id, question_index, answer, is_correct, points, time_taken) 
         VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [matchId, userId, questionIndex, answer, isCorrect ? 1 : 0, points, timeTaken]
      );
    } catch (err) {
      console.error('DB Error:', err);
    }
  });
  
  // Update player progress
  socket.on('player_progress', (data) => {
    const { matchId, currentQuestion, score } = data;
    const userId = socket.userId;
    
    // Broadcast to other players
    socket.to(`match_${matchId}`).emit('opponent_progress', {
      userId,
      currentQuestion,
      score
    });
  });
  
  // Game complete
  socket.on('game_complete', async (data) => {
    const { matchId, winnerId, finalScores } = data;
    
    // Broadcast final results
    io.to(`match_${matchId}`).emit('game_ended', {
      winnerId,
      finalScores
    });
    
    // Update database
    try {
      await db.query(
        'UPDATE multiplayer_matches SET status = ?, winner_id = ?, completed_at = NOW() WHERE id = ?',
        ['completed', winnerId, matchId]
      );
    } catch (err) {
      console.error('DB Error:', err);
    }
    
    // Clean up
    activeMatches.delete(matchId);
    console.log(`🏁 Match ${matchId} completed`);
  });
  
  // Spectate match
  socket.on('spectate', (data) => {
    const { matchId } = data;
    
    socket.join(`match_${matchId}_spectators`);
    
    const match = activeMatches.get(matchId);
    if (match) {
      socket.emit('spectate_started', {
        matchId,
        players: match.players,
        scores: match.scores
      });
    }
  });
  
  // Disconnect
  socket.on('disconnect', () => {
    const userId = socket.userId;
    
    if (userId) {
      userSockets.delete(userId);
      
      // Remove from active matches
      activeMatches.forEach((match, matchId) => {
        const index = match.players.indexOf(userId);
        if (index > -1) {
          match.players.splice(index, 1);
          
          // Notify other players
          io.to(`match_${matchId}`).emit('player_left', { userId });
        }
      });
    }
    
    console.log('🔌 Disconnected:', socket.id);
  });
});

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({
    status: 'running',
    activeMatches: activeMatches.size,
    connectedUsers: userSockets.size,
    timestamp: Date.now()
  });
});

// Start server
const PORT = process.env.PORT || 3000;

connectDB().then(() => {
  httpServer.listen(PORT, () => {
    console.log(`🚀 WebSocket server running on port ${PORT}`);
    console.log(`📡 Socket.IO endpoint: http://localhost:${PORT}`);
  });
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM received, closing server...');
  httpServer.close(() => {
    console.log('Server closed');
    process.exit(0);
  });
});
