-- ============================================
-- StudyPanda Multiplayer Complete Database Schema
-- All multiplayer tables in one file
-- Run this in phpMyAdmin to set up multiplayer
-- ============================================

-- ============================================
-- PART 1: CORE MULTIPLAYER TABLES
-- ============================================

-- Multiplayer matches table
CREATE TABLE IF NOT EXISTS multiplayer_matches (
    id INT PRIMARY KEY AUTO_INCREMENT,
    creator_id INT NOT NULL,
    opponent_id INT DEFAULT NULL,
    game_type ENUM('math_race', 'quiz_battle', 'puzzle_tag', 'word_blast') NOT NULL,
    entry_fee INT DEFAULT 0,
    user_class INT DEFAULT 5,
    questions JSON DEFAULT NULL,
    current_question INT DEFAULT 0,
    creator_score INT DEFAULT 0,
    opponent_score INT DEFAULT 0,
    winner_id INT DEFAULT NULL,
    status ENUM('waiting', 'active', 'bot_match', 'completed', 'cancelled') DEFAULT 'waiting',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    started_at TIMESTAMP NULL,
    completed_at TIMESTAMP NULL,
    FOREIGN KEY (creator_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (opponent_id) REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (winner_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_status (status),
    INDEX idx_game_type (game_type),
    INDEX idx_creator (creator_id),
    INDEX idx_waiting (status, game_type, user_class, created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Match answers table (tracks individual answers)
CREATE TABLE IF NOT EXISTS match_answers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    match_id INT NOT NULL,
    user_id INT NOT NULL,
    question_index INT NOT NULL,
    answer VARCHAR(255) NOT NULL,
    is_correct BOOLEAN DEFAULT FALSE,
    points INT DEFAULT 0,
    time_taken FLOAT DEFAULT 0,
    answered_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (match_id) REFERENCES multiplayer_matches(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_question (match_id, user_id, question_index),
    INDEX idx_match (match_id),
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Multiplayer stats (per user stats for each game type)
CREATE TABLE IF NOT EXISTS multiplayer_stats (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    game_type ENUM('math_race', 'quiz_battle', 'puzzle_tag', 'word_blast') NOT NULL,
    games_played INT DEFAULT 0,
    games_won INT DEFAULT 0,
    games_lost INT DEFAULT 0,
    total_score INT DEFAULT 0,
    best_score INT DEFAULT 0,
    win_streak INT DEFAULT 0,
    best_win_streak INT DEFAULT 0,
    coins_earned INT DEFAULT 0,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_game (user_id, game_type),
    INDEX idx_user (user_id),
    INDEX idx_wins (games_won DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Matchmaking queue
CREATE TABLE IF NOT EXISTS matchmaking_queue (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    game_type ENUM('math_race', 'quiz_battle', 'puzzle_tag', 'word_blast') NOT NULL,
    user_class INT DEFAULT 5,
    entry_fee INT DEFAULT 0,
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP NOT NULL,
    status ENUM('waiting', 'matched', 'expired') DEFAULT 'waiting',
    matched_match_id INT DEFAULT NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_waiting (status, game_type, user_class),
    INDEX idx_user (user_id),
    INDEX idx_expires (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- PART 2: SKILL RATING TABLES (ELO)
-- ============================================

-- Skill rating table (ELO-based)
CREATE TABLE IF NOT EXISTS skill_ratings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    game_type ENUM('math_race', 'quiz_battle', 'puzzle_tag', 'word_blast') NOT NULL,
    rating INT DEFAULT 1000,
    games_played INT DEFAULT 0,
    wins INT DEFAULT 0,
    losses INT DEFAULT 0,
    streak INT DEFAULT 0,
    highest_rating INT DEFAULT 1000,
    last_game_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_game (user_id, game_type),
    INDEX idx_rating (game_type, rating DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Skill tiers for matchmaking
CREATE TABLE IF NOT EXISTS skill_tiers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(50) NOT NULL,
    emoji VARCHAR(20) NOT NULL,
    min_rating INT NOT NULL,
    max_rating INT NOT NULL,
    color VARCHAR(7) NOT NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Insert default skill tiers
INSERT INTO skill_tiers (name, emoji, min_rating, max_rating, color) VALUES
('Bronze', '🥉', 0, 999, '#CD7F32'),
('Silver', '🥈', 1000, 1199, '#C0C0C0'),
('Gold', '🥇', 1200, 1399, '#FFD700'),
('Platinum', '💎', 1400, 1599, '#E5E4E2'),
('Diamond', '💠', 1600, 1799, '#B9F2FF'),
('Master', '🏆', 1800, 1999, '#FF6B6B'),
('Grandmaster', '👑', 2000, 9999, '#FF00FF');

-- ============================================
-- PART 3: TOURNAMENT TABLES
-- ============================================

-- Tournaments table
CREATE TABLE IF NOT EXISTS tournaments (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    game_type ENUM('math_race', 'quiz_battle', 'puzzle_tag', 'word_blast') NOT NULL,
    entry_fee INT DEFAULT 0,
    prize_pool INT DEFAULT 0,
    max_players INT DEFAULT 8,
    creator_id INT NOT NULL,
    winner_id INT DEFAULT NULL,
    start_time TIMESTAMP NULL,
    end_time TIMESTAMP NULL,
    status ENUM('registration', 'active', 'completed', 'cancelled') DEFAULT 'registration',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (creator_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (winner_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_status (status),
    INDEX idx_game_type (game_type),
    INDEX idx_start (start_time)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Tournament players table
CREATE TABLE IF NOT EXISTS tournament_players (
    id INT PRIMARY KEY AUTO_INCREMENT,
    tournament_id INT NOT NULL,
    user_id INT NOT NULL,
    seed INT DEFAULT 0,
    status ENUM('active', 'eliminated', 'winner') DEFAULT 'active',
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (tournament_id) REFERENCES tournaments(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_player (tournament_id, user_id),
    INDEX idx_tournament (tournament_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Tournament matches (bracket)
CREATE TABLE IF NOT EXISTS tournament_matches (
    id INT PRIMARY KEY AUTO_INCREMENT,
    tournament_id INT NOT NULL,
    round INT NOT NULL,
    match_number INT NOT NULL,
    player1_id INT DEFAULT NULL,
    player2_id INT DEFAULT NULL,
    winner_id INT DEFAULT NULL,
    player1_score INT DEFAULT 0,
    player2_score INT DEFAULT 0,
    match_id INT DEFAULT NULL,
    status ENUM('pending', 'active', 'completed') DEFAULT 'pending',
    scheduled_at TIMESTAMP NULL,
    completed_at TIMESTAMP NULL,
    FOREIGN KEY (tournament_id) REFERENCES tournaments(id) ON DELETE CASCADE,
    FOREIGN KEY (player1_id) REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (player2_id) REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (winner_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_tournament_round (tournament_id, round)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- PART 4: SPECTATOR & REPLAY TABLES
-- ============================================

-- Spectator sessions
CREATE TABLE IF NOT EXISTS spectator_sessions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    match_id INT NOT NULL,
    user_id INT NOT NULL,
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    left_at TIMESTAMP NULL,
    FOREIGN KEY (match_id) REFERENCES multiplayer_matches(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_match (match_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- Match replay data (for spectators to catch up)
CREATE TABLE IF NOT EXISTS match_replays (
    id INT PRIMARY KEY AUTO_INCREMENT,
    match_id INT NOT NULL,
    event_type VARCHAR(50) NOT NULL,
    event_data JSON NOT NULL,
    event_time TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (match_id) REFERENCES multiplayer_matches(id) ON DELETE CASCADE,
    INDEX idx_match_time (match_id, event_time)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- CLEANUP QUERIES (Run via cron job)
-- ============================================
-- DELETE FROM multiplayer_matches WHERE status = 'waiting' AND created_at < DATE_SUB(NOW(), INTERVAL 5 MINUTE);
-- DELETE FROM matchmaking_queue WHERE status = 'waiting' AND expires_at < NOW();
