<?php
/**
 * Skill-Based Matchmaking API
 * Uses ELO rating system for fair matches
 */

require_once '../config.php';

function handleSkillMatch($action, $method) {
    switch ($action) {
        case 'find':
            if ($method === 'POST') findSkillMatch();
            break;
        case 'get_rating':
            if ($method === 'GET') getSkillRating();
            break;
        case 'update_rating':
            if ($method === 'POST') updateRating();
            break;
        case 'get_tier':
            if ($method === 'GET') getSkillTier();
            break;
        case 'leaderboard':
            if ($method === 'GET') getSkillLeaderboard();
            break;
        default:
            Response::error('Invalid action', 400);
    }
}

/**
 * Find match with similar skill level
 */
function findSkillMatch() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $gameType = Request::post('game_type');
    $entryFee = Request::post('entry_fee', 0);
    
    // Get user's skill rating
    $stmt = $db->prepare("
        SELECT rating FROM skill_ratings 
        WHERE user_id = ? AND game_type = ?
    ");
    $stmt->execute([$user['user_id'], $gameType]);
    $userRating = $stmt->fetch()['rating'] ?? 1000;
    
    // Define skill range (±200 rating initially, expands over time)
    $ratingRange = 200;
    
    // Look for match in similar skill range
    $stmt = $db->prepare("
        SELECT m.*, sr.rating as creator_rating, u.name as creator_name, u.avatar as creator_avatar
        FROM multiplayer_matches m
        JOIN skill_ratings sr ON sr.user_id = m.creator_id AND sr.game_type = m.game_type
        JOIN users u ON m.creator_id = u.id
        WHERE m.game_type = ?
          AND m.status = 'waiting'
          AND m.creator_id != ?
          AND sr.rating BETWEEN ? AND ?
          AND m.created_at > DATE_SUB(NOW(), INTERVAL 30 SECOND)
        ORDER BY ABS(sr.rating - ?) ASC
        LIMIT 1
    ");
    $stmt->execute([
        $gameType, 
        $user['user_id'], 
        $userRating - $ratingRange, 
        $userRating + $ratingRange,
        $userRating
    ]);
    $existingMatch = $stmt->fetch();
    
    if ($existingMatch) {
        // Join match
        $stmt = $db->prepare("
            UPDATE multiplayer_matches 
            SET opponent_id = ?, status = 'active', started_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$user['user_id'], $existingMatch['id']]);
        
        // Deduct coins
        if ($entryFee > 0) {
            $db->prepare("UPDATE user_stats SET coins = coins - ? WHERE user_id = ?")
               ->execute([$entryFee, $user['user_id']]);
        }
        
        Response::success([
            'match_id' => $existingMatch['id'],
            'status' => 'matched',
            'opponent' => [
                'id' => $existingMatch['creator_id'],
                'name' => $existingMatch['creator_name'],
                'avatar' => $existingMatch['creator_avatar'],
                'rating' => $existingMatch['creator_rating']
            ],
            'user_rating' => $userRating,
            'skill_difference' => abs($existingMatch['creator_rating'] - $userRating)
        ]);
    } else {
        // Create new match
        $stmt = $db->prepare("
            INSERT INTO multiplayer_matches 
            (creator_id, game_type, entry_fee, user_class, status, created_at)
            VALUES (?, ?, ?, ?, 'waiting', NOW())
        ");
        $stmt->execute([$user['user_id'], $gameType, $entryFee, 5]);
        $matchId = $db->lastInsertId();
        
        // Deduct coins
        if ($entryFee > 0) {
            $db->prepare("UPDATE user_stats SET coins = coins - ? WHERE user_id = ?")
               ->execute([$entryFee, $user['user_id']]);
        }
        
        Response::success([
            'match_id' => $matchId,
            'status' => 'waiting',
            'user_rating' => $userRating,
            'timeout_seconds' => 30
        ]);
    }
}

/**
 * Get user's skill rating
 */
function getSkillRating() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $gameType = Request::get('game_type');
    
    // Get or create rating
    $stmt = $db->prepare("
        SELECT sr.*, st.name as tier_name, st.emoji as tier_emoji, st.color as tier_color
        FROM skill_ratings sr
        LEFT JOIN skill_tiers st ON sr.rating BETWEEN st.min_rating AND st.max_rating
        WHERE sr.user_id = ? AND sr.game_type = ?
    ");
    $stmt->execute([$user['user_id'], $gameType]);
    $rating = $stmt->fetch();
    
    if (!$rating) {
        // Create default rating
        $db->prepare("
            INSERT INTO skill_ratings (user_id, game_type, rating)
            VALUES (?, ?, 1000)
        ")->execute([$user['user_id'], $gameType]);
        
        $rating = [
            'rating' => 1000,
            'games_played' => 0,
            'wins' => 0,
            'losses' => 0,
            'tier_name' => 'Silver',
            'tier_emoji' => '🥈',
            'tier_color' => '#C0C0C0'
        ];
    }
    
    Response::success($rating);
}

/**
 * Update rating after match (ELO calculation)
 */
function updateRating() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $matchId = Request::post('match_id');
    $winnerId = Request::post('winner_id');
    
    // Get match
    $stmt = $db->prepare("SELECT * FROM multiplayer_matches WHERE id = ?");
    $stmt->execute([$matchId]);
    $match = $stmt->fetch();
    
    if (!$match) {
        Response::error('Match not found', 404);
    }
    
    $player1 = $match['creator_id'];
    $player2 = $match['opponent_id'];
    $gameType = $match['game_type'];
    
    if (!$player2) {
        // Bot match - smaller rating change
        $ratingChange = ($winnerId == $player1) ? 5 : -5;
        updatePlayerRating($player1, $gameType, $ratingChange, $winnerId == $player1);
        Response::success(['rating_change' => $ratingChange]);
        return;
    }
    
    // Get ratings
    $rating1 = getPlayerRating($player1, $gameType);
    $rating2 = getPlayerRating($player2, $gameType);
    
    // ELO calculation
    $k = 32; // K-factor
    
    $expected1 = 1 / (1 + pow(10, ($rating2 - $rating1) / 400));
    $expected2 = 1 / (1 + pow(10, ($rating1 - $rating2) / 400));
    
    $score1 = ($winnerId == $player1) ? 1 : 0;
    $score2 = ($winnerId == $player2) ? 1 : 0;
    
    $change1 = round($k * ($score1 - $expected1));
    $change2 = round($k * ($score2 - $expected2));
    
    // Update ratings
    updatePlayerRating($player1, $gameType, $change1, $winnerId == $player1);
    updatePlayerRating($player2, $gameType, $change2, $winnerId == $player2);
    
    Response::success([
        'player1_change' => $change1,
        'player2_change' => $change2,
        'player1_new_rating' => $rating1 + $change1,
        'player2_new_rating' => $rating2 + $change2
    ]);
}

function getPlayerRating($userId, $gameType) {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("
        SELECT rating FROM skill_ratings 
        WHERE user_id = ? AND game_type = ?
    ");
    $stmt->execute([$userId, $gameType]);
    $result = $stmt->fetch();
    return $result['rating'] ?? 1000;
}

function updatePlayerRating($userId, $gameType, $change, $isWin) {
    $db = Database::getInstance()->getConnection();
    
    // Ensure rating exists
    $db->prepare("
        INSERT IGNORE INTO skill_ratings (user_id, game_type, rating)
        VALUES (?, ?, 1000)
    ")->execute([$userId, $gameType]);
    
    // Update rating
    $newStreak = $isWin ? 'streak + 1' : '0';
    $db->prepare("
        UPDATE skill_ratings 
        SET rating = GREATEST(0, rating + ?),
            games_played = games_played + 1,
            wins = wins + ?,
            losses = losses + ?,
            streak = $newStreak,
            highest_rating = GREATEST(highest_rating, rating + ?),
            last_game_at = NOW()
        WHERE user_id = ? AND game_type = ?
    ")->execute([$change, $isWin ? 1 : 0, $isWin ? 0 : 1, $change, $userId, $gameType]);
}

/**
 * Get user's skill tier
 */
function getSkillTier() {
    $db = Database::getInstance()->getConnection();
    $rating = Request::get('rating', 1000);
    
    $stmt = $db->prepare("
        SELECT * FROM skill_tiers 
        WHERE ? BETWEEN min_rating AND max_rating
    ");
    $stmt->execute([$rating]);
    $tier = $stmt->fetch();
    
    Response::success($tier);
}

/**
 * Get skill leaderboard
 */
function getSkillLeaderboard() {
    $db = Database::getInstance()->getConnection();
    $gameType = Request::get('game_type');
    $limit = Request::get('limit', 50);
    
    $stmt = $db->prepare("
        SELECT sr.*, u.name, u.avatar,
               st.name as tier_name, st.emoji as tier_emoji, st.color as tier_color,
               @rank := @rank + 1 as rank_position
        FROM skill_ratings sr
        JOIN users u ON sr.user_id = u.id
        LEFT JOIN skill_tiers st ON sr.rating BETWEEN st.min_rating AND st.max_rating
        CROSS JOIN (SELECT @rank := 0) r
        WHERE sr.game_type = ?
          AND sr.games_played >= 5
        ORDER BY sr.rating DESC
        LIMIT ?
    ");
    $stmt->execute([$gameType, $limit]);
    $leaderboard = $stmt->fetchAll();
    
    Response::success(['leaderboard' => $leaderboard]);
}
?>
