<?php
/**
 * Multiplayer Game API Endpoints
 * Handles matchmaking, game state, and results
 */

require_once '../config.php';

function handleMultiplayer($action, $method) {
    switch ($action) {
        case 'find_match':
            if ($method === 'POST') findMatch();
            break;
        case 'check_match':
            if ($method === 'GET') checkMatchStatus();
            break;
        case 'join_match':
            if ($method === 'POST') joinMatch();
            break;
        case 'submit_answer':
            if ($method === 'POST') submitAnswer();
            break;
        case 'get_game_state':
            if ($method === 'GET') getGameState();
            break;
        case 'end_game':
            if ($method === 'POST') endGame();
            break;
        case 'cancel_match':
            if ($method === 'POST') cancelMatch();
            break;
        default:
            Response::error('Invalid action', 400);
    }
}

/**
 * Find or create a match
 * If no opponent found in 30 seconds, returns bot match
 */
function findMatch() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $gameType = Request::post('game_type'); // math_race, quiz_battle, puzzle_tag, word_blast
    $entryFee = Request::post('entry_fee', 0);
    $userClass = Request::post('user_class', 5);
    
    if (!$gameType) {
        Response::error('Game type required', 400);
    }
    
    // Check if user has enough coins
    $stmt = $db->prepare("SELECT coins FROM user_stats WHERE user_id = ?");
    $stmt->execute([$user['user_id']]);
    $stats = $stmt->fetch();
    
    if ($stats['coins'] < $entryFee) {
        Response::error('Insufficient coins', 400);
    }
    
    // Check for existing waiting matches
    $stmt = $db->prepare("
        SELECT m.*, u.name as creator_name, u.avatar as creator_avatar
        FROM multiplayer_matches m
        JOIN users u ON m.creator_id = u.id
        WHERE m.game_type = ?
          AND m.status = 'waiting'
          AND m.creator_id != ?
          AND m.user_class = ?
          AND m.created_at > DATE_SUB(NOW(), INTERVAL 30 SECOND)
        ORDER BY m.created_at ASC
        LIMIT 1
    ");
    $stmt->execute([$gameType, $user['user_id'], $userClass]);
    $existingMatch = $stmt->fetch();
    
    if ($existingMatch) {
        // Join existing match
        $stmt = $db->prepare("
            UPDATE multiplayer_matches 
            SET opponent_id = ?, status = 'active', started_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$user['user_id'], $existingMatch['id']]);
        
        // Deduct entry fee from both players
        if ($entryFee > 0) {
            $db->prepare("UPDATE user_stats SET coins = coins - ? WHERE user_id = ?")
               ->execute([$entryFee, $user['user_id']]);
        }
        
        Response::success([
            'match_id' => $existingMatch['id'],
            'status' => 'matched',
            'is_creator' => false,
            'opponent' => [
                'id' => $existingMatch['creator_id'],
                'name' => $existingMatch['creator_name'],
                'avatar' => $existingMatch['creator_avatar']
            ]
        ], 'Match found!');
    } else {
        // Create new match
        $stmt = $db->prepare("
            INSERT INTO multiplayer_matches 
            (creator_id, game_type, entry_fee, user_class, status, created_at)
            VALUES (?, ?, ?, ?, 'waiting', NOW())
        ");
        $stmt->execute([$user['user_id'], $gameType, $entryFee, $userClass]);
        $matchId = $db->lastInsertId();
        
        // Deduct entry fee
        if ($entryFee > 0) {
            $db->prepare("UPDATE user_stats SET coins = coins - ? WHERE user_id = ?")
               ->execute([$entryFee, $user['user_id']]);
        }
        
        Response::success([
            'match_id' => $matchId,
            'status' => 'waiting',
            'is_creator' => true,
            'timeout_seconds' => 30
        ], 'Waiting for opponent...');
    }
}

/**
 * Check match status (polling endpoint)
 */
function checkMatchStatus() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $matchId = Request::get('match_id');
    
    if (!$matchId) {
        Response::error('Match ID required', 400);
    }
    
    $stmt = $db->prepare("
        SELECT m.*, 
               uc.name as creator_name, uc.avatar as creator_avatar,
               uo.name as opponent_name, uo.avatar as opponent_avatar
        FROM multiplayer_matches m
        LEFT JOIN users uc ON m.creator_id = uc.id
        LEFT JOIN users uo ON m.opponent_id = uo.id
        WHERE m.id = ? AND (m.creator_id = ? OR m.opponent_id = ?)
    ");
    $stmt->execute([$matchId, $user['user_id'], $user['user_id']]);
    $match = $stmt->fetch();
    
    if (!$match) {
        Response::error('Match not found', 404);
    }
    
    // Check if match timed out (30 seconds)
    $createdAt = strtotime($match['created_at']);
    $elapsed = time() - $createdAt;
    
    if ($match['status'] === 'waiting' && $elapsed >= 30) {
        // Update to bot match
        $stmt = $db->prepare("
            UPDATE multiplayer_matches 
            SET status = 'bot_match', started_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$matchId]);
        $match['status'] = 'bot_match';
    }
    
    $opponent = null;
    if ($match['opponent_id']) {
        $isCreator = ($match['creator_id'] == $user['user_id']);
        $opponent = [
            'id' => $isCreator ? $match['opponent_id'] : $match['creator_id'],
            'name' => $isCreator ? $match['opponent_name'] : $match['creator_name'],
            'avatar' => $isCreator ? $match['opponent_avatar'] : $match['creator_avatar']
        ];
    }
    
    Response::success([
        'match_id' => $matchId,
        'status' => $match['status'],
        'game_type' => $match['game_type'],
        'is_creator' => ($match['creator_id'] == $user['user_id']),
        'opponent' => $opponent,
        'elapsed_seconds' => $elapsed,
        'questions' => json_decode($match['questions'] ?? '[]', true),
        'current_question' => $match['current_question'],
        'creator_score' => $match['creator_score'],
        'opponent_score' => $match['opponent_score']
    ]);
}

/**
 * Submit answer during game
 */
function submitAnswer() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $matchId = Request::post('match_id');
    $questionIndex = Request::post('question_index');
    $answer = Request::post('answer');
    $timeTaken = Request::post('time_taken', 0);
    
    // Get match
    $stmt = $db->prepare("SELECT * FROM multiplayer_matches WHERE id = ?");
    $stmt->execute([$matchId]);
    $match = $stmt->fetch();
    
    if (!$match) {
        Response::error('Match not found', 404);
    }
    
    // Check if user is in this match
    $isCreator = ($match['creator_id'] == $user['user_id']);
    $isOpponent = ($match['opponent_id'] == $user['user_id']);
    
    if (!$isCreator && !$isOpponent) {
        Response::error('Not in this match', 403);
    }
    
    // Get questions
    $questions = json_decode($match['questions'], true);
    if ($questionIndex >= count($questions)) {
        Response::error('Invalid question index', 400);
    }
    
    $question = $questions[$questionIndex];
    $isCorrect = (strtolower(trim($answer)) == strtolower(trim($question['answer'])));
    
    // Calculate points (speed bonus)
    $points = 0;
    if ($isCorrect) {
        $basePoints = 100;
        $speedBonus = max(0, 50 - ($timeTaken * 5)); // Max 50 bonus
        $points = $basePoints + $speedBonus;
    }
    
    // Record answer
    $stmt = $db->prepare("
        INSERT INTO match_answers 
        (match_id, user_id, question_index, answer, is_correct, points, time_taken, answered_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ON DUPLICATE KEY UPDATE answer = VALUES(answer), is_correct = VALUES(is_correct), 
                                points = VALUES(points), time_taken = VALUES(time_taken)
    ");
    $stmt->execute([$matchId, $user['user_id'], $questionIndex, $answer, $isCorrect ? 1 : 0, $points, $timeTaken]);
    
    // Update match scores
    if ($isCreator) {
        $db->prepare("UPDATE multiplayer_matches SET creator_score = creator_score + ? WHERE id = ?")
           ->execute([$points, $matchId]);
    } else {
        $db->prepare("UPDATE multiplayer_matches SET opponent_score = opponent_score + ? WHERE id = ?")
           ->execute([$points, $matchId]);
    }
    
    Response::success([
        'is_correct' => $isCorrect,
        'correct_answer' => $question['answer'],
        'points_earned' => $points,
        'explanation' => $question['explanation'] ?? null
    ]);
}

/**
 * Get current game state
 */
function getGameState() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $matchId = Request::get('match_id');
    
    $stmt = $db->prepare("
        SELECT m.*, 
               uc.name as creator_name, uc.avatar as creator_avatar,
               uo.name as opponent_name, uo.avatar as opponent_avatar
        FROM multiplayer_matches m
        LEFT JOIN users uc ON m.creator_id = uc.id
        LEFT JOIN users uo ON m.opponent_id = uo.id
        WHERE m.id = ?
    ");
    $stmt->execute([$matchId]);
    $match = $stmt->fetch();
    
    if (!$match) {
        Response::error('Match not found', 404);
    }
    
    // Get all answers for this match
    $stmt = $db->prepare("
        SELECT * FROM match_answers WHERE match_id = ? ORDER BY question_index
    ");
    $stmt->execute([$matchId]);
    $answers = $stmt->fetchAll();
    
    $isCreator = ($match['creator_id'] == $user['user_id']);
    
    Response::success([
        'match_id' => $matchId,
        'status' => $match['status'],
        'game_type' => $match['game_type'],
        'questions' => json_decode($match['questions'] ?? '[]', true),
        'current_question' => $match['current_question'],
        'my_score' => $isCreator ? $match['creator_score'] : $match['opponent_score'],
        'opponent_score' => $isCreator ? $match['opponent_score'] : $match['creator_score'],
        'answers' => $answers
    ]);
}

/**
 * End game and calculate rewards
 */
function endGame() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $matchId = Request::post('match_id');
    
    $stmt = $db->prepare("SELECT * FROM multiplayer_matches WHERE id = ?");
    $stmt->execute([$matchId]);
    $match = $stmt->fetch();
    
    if (!$match || $match['status'] === 'completed') {
        Response::error('Invalid match', 400);
    }
    
    $isCreator = ($match['creator_id'] == $user['user_id']);
    $creatorScore = $match['creator_score'];
    $opponentScore = $match['opponent_score'];
    
    // Determine winner
    $winnerId = null;
    if ($creatorScore > $opponentScore) {
        $winnerId = $match['creator_id'];
    } elseif ($opponentScore > $creatorScore) {
        $winnerId = $match['opponent_id'];
    }
    
    // Calculate rewards based on game type
    $entryFee = $match['entry_fee'];
    $winReward = 0;
    $loseReward = 0;
    
    switch ($match['game_type']) {
        case 'math_race':
            $winReward = 25;
            $loseReward = 5;
            break;
        case 'quiz_battle':
            $winReward = 30;
            $loseReward = 5;
            break;
        case 'puzzle_tag':
            $winReward = 15;
            $loseReward = 2;
            break;
        case 'word_blast':
            $winReward = 20;
            $loseReward = 3;
            break;
    }
    
    // Award coins to players
    if ($winnerId) {
        $loserId = ($winnerId == $match['creator_id']) ? $match['opponent_id'] : $match['creator_id'];
        
        // Winner gets win reward
        $db->prepare("UPDATE user_stats SET coins = coins + ? WHERE user_id = ?")
           ->execute([$winReward, $winnerId]);
        
        // Loser gets participation reward (only if real player)
        if ($loserId && $match['status'] !== 'bot_match') {
            $db->prepare("UPDATE user_stats SET coins = coins + ? WHERE user_id = ?")
               ->execute([$loseReward, $loserId]);
        }
    }
    
    // Update match status
    $stmt = $db->prepare("
        UPDATE multiplayer_matches 
        SET status = 'completed', winner_id = ?, completed_at = NOW()
        WHERE id = ?
    ");
    $stmt->execute([$winnerId, $matchId]);
    
    $isWinner = ($winnerId == $user['user_id']);
    
    Response::success([
        'winner_id' => $winnerId,
        'is_winner' => $isWinner,
        'coins_earned' => $isWinner ? $winReward : $loseReward,
        'my_score' => $isCreator ? $creatorScore : $opponentScore,
        'opponent_score' => $isCreator ? $opponentScore : $creatorScore
    ]);
}

/**
 * Cancel matchmaking
 */
function cancelMatch() {
    $user = JWT::requireAuth();
    $db = Database::getInstance()->getConnection();
    
    $matchId = Request::post('match_id');
    
    $stmt = $db->prepare("
        SELECT * FROM multiplayer_matches 
        WHERE id = ? AND creator_id = ? AND status = 'waiting'
    ");
    $stmt->execute([$matchId, $user['user_id']]);
    $match = $stmt->fetch();
    
    if (!$match) {
        Response::error('Match not found or cannot be cancelled', 400);
    }
    
    // Refund entry fee
    if ($match['entry_fee'] > 0) {
        $db->prepare("UPDATE user_stats SET coins = coins + ? WHERE user_id = ?")
           ->execute([$match['entry_fee'], $user['user_id']]);
    }
    
    // Delete match
    $db->prepare("DELETE FROM multiplayer_matches WHERE id = ?")->execute([$matchId]);
    
    Response::success(['refunded' => $match['entry_fee']], 'Match cancelled');
}
?>
