<?php
/**
 * Leaderboard API Endpoints
 */

function handleLeaderboard($action, $method) {
    if ($method !== 'GET') {
        Response::error('Method not allowed', 405);
    }
    
    switch ($action) {
        case 'global':
            getGlobalLeaderboard();
            break;
        case 'friends':
            $user = JWT::requireAuth();
            getFriendsLeaderboard($user);
            break;
        case 'weekly':
            getWeeklyLeaderboard();
            break;
        case 'daily':
            getDailyLeaderboard();
            break;
        default:
            getGlobalLeaderboard();
    }
}

function getGlobalLeaderboard() {
    $db = Database::getInstance()->getConnection();
    $limit = Request::get('limit', 50);
    $offset = Request::get('offset', 0);
    
    $stmt = $db->prepare("
        SELECT u.id, u.name, u.avatar, us.xp, us.level,
               @rank := @rank + 1 as rank_position
        FROM users u
        INNER JOIN user_stats us ON u.id = us.user_id
        CROSS JOIN (SELECT @rank := ?) r
        WHERE u.is_active = 1
        ORDER BY us.xp DESC
        LIMIT ? OFFSET ?
    ");
    $stmt->execute([$offset, $limit, $offset]);
    $leaderboard = $stmt->fetchAll();
    
    // Get total count
    $stmt = $db->prepare("SELECT COUNT(*) as total FROM users WHERE is_active = 1");
    $stmt->execute();
    $total = $stmt->fetch()['total'];
    
    Response::success([
        'leaderboard' => $leaderboard,
        'total' => $total,
        'limit' => $limit,
        'offset' => $offset
    ]);
}

function getFriendsLeaderboard($authUser) {
    $db = Database::getInstance()->getConnection();
    
    $stmt = $db->prepare("
        SELECT u.id, u.name, u.avatar, us.xp, us.level
        FROM users u
        INNER JOIN user_stats us ON u.id = us.user_id
        WHERE u.id = ?
           OR u.id IN (
               SELECT friend_id FROM friends WHERE user_id = ? AND status = 'accepted'
               UNION
               SELECT user_id FROM friends WHERE friend_id = ? AND status = 'accepted'
           )
        ORDER BY us.xp DESC
        LIMIT 50
    ");
    $stmt->execute([$authUser['user_id'], $authUser['user_id'], $authUser['user_id']]);
    $leaderboard = $stmt->fetchAll();
    
    // Add rank
    $rank = 1;
    foreach ($leaderboard as &$player) {
        $player['rank_position'] = $rank++;
        $player['is_me'] = ($player['id'] == $authUser['user_id']);
    }
    
    Response::success(['leaderboard' => $leaderboard]);
}

function getWeeklyLeaderboard() {
    $db = Database::getInstance()->getConnection();
    $startOfWeek = date('Y-m-d', strtotime('monday this week'));
    
    $stmt = $db->prepare("
        SELECT u.id, u.name, u.avatar, 
               COALESCE(SUM(al.xp_earned), 0) as weekly_xp,
               us.level
        FROM users u
        INNER JOIN user_stats us ON u.id = us.user_id
        LEFT JOIN (
            SELECT user_id, 
                   CAST(JSON_EXTRACT(action_data, '$.xp') AS UNSIGNED) as xp_earned
            FROM activity_log 
            WHERE action_type = 'game_complete' 
              AND created_at >= ?
        ) al ON u.id = al.user_id
        WHERE u.is_active = 1
        GROUP BY u.id
        ORDER BY weekly_xp DESC
        LIMIT 50
    ");
    $stmt->execute([$startOfWeek]);
    $leaderboard = $stmt->fetchAll();
    
    // Add rank
    $rank = 1;
    foreach ($leaderboard as &$player) {
        $player['rank_position'] = $rank++;
    }
    
    Response::success(['leaderboard' => $leaderboard, 'period' => 'weekly']);
}

function getDailyLeaderboard() {
    $db = Database::getInstance()->getConnection();
    $today = date('Y-m-d');
    
    $stmt = $db->prepare("
        SELECT u.id, u.name, u.avatar,
               COALESCE(SUM(al.xp_earned), 0) as daily_xp,
               us.level
        FROM users u
        INNER JOIN user_stats us ON u.id = us.user_id
        LEFT JOIN (
            SELECT user_id,
                   CAST(JSON_EXTRACT(action_data, '$.xp') AS UNSIGNED) as xp_earned
            FROM activity_log
            WHERE action_type = 'game_complete'
              AND DATE(created_at) = ?
        ) al ON u.id = al.user_id
        WHERE u.is_active = 1
        GROUP BY u.id
        ORDER BY daily_xp DESC
        LIMIT 50
    ");
    $stmt->execute([$today]);
    $leaderboard = $stmt->fetchAll();
    
    // Add rank
    $rank = 1;
    foreach ($leaderboard as &$player) {
        $player['rank_position'] = $rank++;
    }
    
    Response::success(['leaderboard' => $leaderboard, 'period' => 'daily']);
}
?>
